/*
 * Copyright 2004-2012 ICEsoft Technologies Canada Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the
 * License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an "AS
 * IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.icefaces.impl.util;

import org.icefaces.impl.event.UIOutputWriter;
import org.icefaces.util.EnvUtils;

import javax.faces.component.UIComponent;
import javax.faces.component.UIOutput;
import javax.faces.component.UIViewRoot;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.event.AbortProcessingException;
import javax.faces.event.SystemEvent;
import javax.faces.event.SystemEventListener;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * The purpose of this class is to add a script to the body of a portlet.  The script is designed to communication with
 * Liferay's client-side session timer with each Ajax request sent by ICEfaces.  In this way, the timer is kept in sync
 * for requests that may not be normally handled by the portal.
 */
public class LiferayListener implements SystemEventListener {

    private final static Logger log = Logger.getLogger(LiferayListener.class.getName());

    public void processEvent(SystemEvent systemEvent) throws AbortProcessingException {
        final FacesContext fc = FacesContext.getCurrentInstance();

        if (!EnvUtils.isICEfacesView(fc)) {
            //If ICEfaces is not configured for this view, we don't need to process this event.
            return;
        }

        if (!EnvUtils.instanceofPortletRequest(fc.getExternalContext().getRequest())) {
            //If we're not running in a portlet, we don't need to process this event.
            return;
        }

        if (EnvUtils.isLiferay()) {
            //This script is only intended for running on Liferay
            UIViewRoot root = fc.getViewRoot();
            root.addComponentResource(fc, getLiferayScriptComponent(fc), "body");
        }

    }

    public boolean isListenerForSource(Object o) {
        return true;
    }

    public static UIComponent getLiferayScriptComponent(FacesContext fc) {

        final String liferayScript = LiferayUtil.getLiferayScript(fc);
        if (log.isLoggable(Level.FINE)) {
            log.log(Level.FINE, "adding script: \n" + liferayScript);
        }

        UIOutput liferayScriptComponent = new UIOutputWriter() {

            public void encode(ResponseWriter writer, FacesContext context) throws IOException {
                writer.startElement("script", this);
                writer.write(liferayScript);
                writer.endElement("script");
            }
        };

        liferayScriptComponent.setTransient(true);
        if (log.isLoggable(Level.FINE)) {
            log.log(Level.FINE, "added Liferay script component to body\n" + liferayScript);
        }
        return liferayScriptComponent;
    }
}
