/*
 * Copyright 2004-2012 ICEsoft Technologies Canada Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the
 * License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an "AS
 * IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.icefaces.impl.util;

import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import java.util.logging.Level;
import java.util.logging.Logger;

public class LiferayUtil {

    private final static Logger log = Logger.getLogger(LiferayUtil.class.getName());

    /**
     * Utility method for generating the script after querying the context parameters or using the default values as
     * appropriate.
     *
     * @param fc The current FacesContext
     * @return The generated contents of the script
     */
    public static String getLiferayScript(FacesContext fc) {

        //The problematic behaviour was fixed and the API altered as of Liferay EE 6.0.13
        //and Liferay CE 6.1.  So the script that we deliver is slightly different depending
        //on the version of Liferay in use.
        boolean liferayFixed = getContextParameterAsBoolean(fc, "liferayFixed", false);

        //By default we'll call Liferay.Session.extend() method as it is more reliable
        //but it does mean a separate request is made to the server.  The setCookie
        //call is more problematic but does not trigger a request to the server.
        boolean useExtendAPI = getContextParameterAsBoolean(fc, "useExtendAPI", true);

        //The API used to register a callback function before we submit something changed
        //between ICEfaces 1.x and 2.x+ but the older one is still supported.
        boolean useICEfaces1API = getContextParameterAsBoolean(fc, "useICEfaces1API", true);

        //Whether to include console.log statements for debugging on the client.
        boolean includeLogging = getContextParameterAsBoolean(fc, "includeLogging", false);

        String liferayScript = getLiferayScript(useExtendAPI, liferayFixed, useICEfaces1API, includeLogging);
        if (log.isLoggable(Level.FINE)) {
            log.log(Level.FINE, "Liferay script parameters:" +
                    "\n  liferayFixed   : " + liferayFixed +
                    "\n  useExtendAPI   : " + useExtendAPI +
                    "\n  useICEfaces1API: " + useICEfaces1API +
                    "\n  includeLogging : " + includeLogging);
        }

        return liferayScript;
    }

    /**
     * This returns a snippet of JavaScript designed to help keep the Liferay client-side session from timing out. Ajax
     * requests that don't go through the portal are missed by Liferay and this script should "tickle" the session with
     * each Ajax request.
     *
     * @param useExtendAPI    If true, indicates that you want to extend the session as well as make a request to the
     *                        server. Determines whether or not to use Liferay's extend() method which triggers a
     *                        request. This is more reliable for versions of Liferay before the fix was made. If false,
     *                        then the appropriate API to extend the session is used which does not trigger a request
     *                        to
     *                        the server. This works together with the liferayFixed parameter.
     * @param liferayFixed    If true, indicates we are running a version of Liferay that has been fixed and the newer
     *                        client-side API is used to extend the session. If false,
     *                        the older API is used. This works
     *                        together with the useExtendAPI parameter.
     * @param useICEfaces1API If true we use the onSubmitSend() callback which is still supported in newer versions of
     *                        ICEfaces.  If false, onBeforeSubmit() is used.
     * @param includeLogging  If true, console.log() statements are added to help with client-side debugging.
     * @return The generated contents of the script.
     */
    public static String getLiferayScript(boolean useExtendAPI,
                                          boolean liferayFixed,
                                          boolean useICEfaces1API,
                                          boolean includeLogging) {
        StringBuilder buff = new StringBuilder();
        buff.append("if(!extendLiferayClientTimer){");
        if(includeLogging){
            buff.append(" console.log('registering extendLiferayClientTimer() callback');");
        }
        buff.append(" var extendLiferayClientTimer = function() {");
        buff.append(" if(Liferay.Session){");

        // The extend() function works in all versions of Liferay but does make an
        // extra call (in addition to the Ajax request) to the server to extend the
        // session.
        if (useExtendAPI) {
            if(includeLogging){
                buff.append(" console.log('preparing to call Liferay.Session.extend()');");
            }
            buff.append(" Liferay.Session.extend();");
        } else {
            // To avoid a call to the server, you can use these alternate methods but
            // setCookie is highly unreliable and prompted the fix as per
            // http://issues.liferay.com/browse/LPS-17624
            if (liferayFixed) {
                if(includeLogging){
                    buff.append(" console.log('preparing to call Liferay.Session.set');");
                }
                buff.append(" Liferay.Session.set('sessionState', 'active')");
            } else {
                if(includeLogging){
                    buff.append(" console.log('preparing to call Liferay.Session.setCookie()');");
                }
                buff.append(" Liferay.Session.setCookie();");
            }
        }

        buff.append(" }");
        buff.append(" };");
        buff.append(" if (Liferay) {");

        if (useICEfaces1API) {
            buff.append(" ice.onSubmitSend(extendLiferayClientTimer);");
        } else {
            buff.append(" ice.onBeforeSubmit(extendLiferayClientTimer);");
        }

        buff.append(" }");
        buff.append("}");

        return buff.toString();
    }

    private static boolean getContextParameterAsBoolean(FacesContext fc, String name, boolean defaultValue) {
        ExternalContext ec = fc.getExternalContext();
        String paramValue = ec.getInitParameter("org.icefaces.portlet.liferay." + name);
        if (paramValue == null) {
            return defaultValue;
        }
        return Boolean.parseBoolean(paramValue);
    }
}